/*-----------------------------------------------------------------------------
 * EcInterfaceCommon.h
 * Copyright                acontis technologies GmbH, Weingarten, Germany
 * Response                 Stefan Zintgraf
 * Description              EtherCAT Master common definitions for interfaces
 *---------------------------------------------------------------------------*/

#ifndef INC_ECINTERFACECOMMON
#define INC_ECINTERFACECOMMON

#if !(defined __ECESCREG_H__)
#include <EcEscReg.h>
#endif

/*-PACK SETTINGS-------------------------------------------------------------*/
#if defined  __GNUC__   /* GNU */

#define EC_PACKED __attribute__((aligned(4), packed))

#if (defined __RCX__)
#pragma pack (4)
#endif

#elif defined _DIAB_TOOL

#define EC_PACKED __attribute__((aligned(4), packed))

#elif defined _MSC_VER  /* MICROSOFT */

#define EC_PACKED
#pragma pack (push, 4)
#pragma warning(disable:4514)

#elif defined __MET__

#define EC_PACKED
/*#pragma pack (push, 4)*/
#pragma Push_align_members(4)

#else
#error Please check pack settings for this compiler!!!
#endif

/*-DEFINES/MACROS------------------------------------------------------------*/
#define ECAT_DEVICE_NAMESIZE                80

#define SAFE_STRCPY(szTargetString,szSourceString,nMaxSize)     \
    {                                                           \
    OsMemset((szTargetString),0,(nMaxSize));                    \
    OsStrncpy((szTargetString),(szSourceString),(nMaxSize)-1);  \
    }


#define MAX_ERRINFO_STRLEN  8      /* maximum length of error notification info strings */
#define MAX_SHORT_STRLEN    20      /* maximum length of a short info string */
#define MAX_STD_STRLEN      80      /* maximum length of a standard info string */

#define MAX_FILE_NAME_SIZE                  0x0020      /* Max file name length for FOE Download/Upload */

#define MASTER_SLAVE_ID     ((EC_T_DWORD)0x10000)

/* textual description of EtherCat commands */
#define EcatCmdShortText(byCmd)                     \
    ((byCmd)==EC_CMD_TYPE_NOP?"NOP":                \
     ((byCmd)==EC_CMD_TYPE_APRD?"APRD":             \
      ((byCmd)==EC_CMD_TYPE_APWR?"APWR":            \
       ((byCmd)==EC_CMD_TYPE_APRW?"APRW":           \
        ((byCmd)==EC_CMD_TYPE_FPRD?"FPRD":          \
         ((byCmd)==EC_CMD_TYPE_FPWR?"FPWR":         \
          ((byCmd)==EC_CMD_TYPE_FPRW?"FPRW":        \
           ((byCmd)==EC_CMD_TYPE_BRD?"BRD":         \
            ((byCmd)==EC_CMD_TYPE_BWR?"BWR":        \
             ((byCmd)==EC_CMD_TYPE_BRW?"BRW":       \
              ((byCmd)==EC_CMD_TYPE_LRD?"LRD":      \
               ((byCmd)==EC_CMD_TYPE_LWR?"LWR":     \
                ((byCmd)==EC_CMD_TYPE_LRW?"LRW":    \
                 ((byCmd)==EC_CMD_TYPE_ARMW?"ARMW": \
                  "INVALID ECAT CMD VALUE!!!"       \
    ))))))))))))))



typedef enum _EC_T_STATEMACH_RES
{
    eStateMachRes_Pending   =   1,
    eStateMachRes_Done      =   2,
    eStateMachRes_Error     =   4,

    eStateMachRes_BCppDummy =   0xFFFFFFFF,
}   EC_T_STATEMACH_RES;

typedef enum _EC_T_LINKMODE
{        
    EcLinkMode_UNDEFINED = 0,
    EcLinkMode_INTERRUPT = 1,
    EcLinkMode_POLLING   = 2,
    EcLinkMode_RANDOM    = 3,      /* for debug purposes only! */

    /* Borland C++ datatype alignment correction */
    EcLinkMode_BCppDummy   = 0xFFFFFFFF
} EC_T_LINKMODE;

typedef enum EC_MAILBOX_FLAG
{
    EC_MAILBOX_FLAG_SDO_COMPLETE			= 1,
    
    /* Borland C++ datatype alignment correction */
    EC_MAILBOX_FLAG_BCppDummy                               = 0xFFFFFFFF
} EC_MAILBOX_FLAG;


/* EtherCAT master states where cyclic commands can be sent */
typedef enum _EC_T_CYCCMD_STATE
{
    eEcatCycCmdState_ANYOP    = 0,            /*< any of the operational states below */
    eEcatCycCmdState_INIT     = 1,            /*< init */
    eEcatCycCmdState_PREOP    = 2,            /*< pre-operational */
    eEcatCycCmdState_SAFEOP   = 3,            /*< safe operational */
    eEcatCycCmdState_OP       = 4,            /*< operational */

    eEcatCycCmdState_PREOPDC  = 5,            /*< pre-operational, DC commands removed */

    /* Borland C++ datatype alignment correction */
    eEcatCycCmdState_BCppDummy   = 0xFFFFFFFF
} EC_T_MASTER_STATE;
#define EC_NUM_CYCCMD_STATES 6

#define CycCmdStateText(EState)                     \
    ((EState)==eEcatCycCmdState_ANYOP?"ANYOP":      \
     ((EState)==eEcatCycCmdState_INIT?"INIT":     \
      ((EState)==eEcatCycCmdState_PREOP?"PREOP":     \
       ((EState)==eEcatCycCmdState_PREOPDC?"PREOPDC":     \
        ((EState)==eEcatCycCmdState_SAFEOP?"SAFEOP":  \
         ((EState)==eEcatCycCmdState_OP?"OP":         \
          "INVALID STATE VALUE!!!"                    \
    ))))))

#if (eEcatCycCmdState_ANYOP>=EC_NUM_CYCCMD_STATES)
#error eEcatCycCmdState_ANYOP>=EC_NUM_CYCCMD_STATES
#endif
#if (eEcatCycCmdState_PREOP>=EC_NUM_CYCCMD_STATES)
#error eEcatCycCmdState_PREOP>=EC_NUM_CYCCMD_STATES
#endif
#if (eEcatCycCmdState_PREOPDC>=EC_NUM_CYCCMD_STATES)
#error eEcatCycCmdState_PREOPDC>=EC_NUM_CYCCMD_STATES
#endif
#if (eEcatCycCmdState_SAFEOP>=EC_NUM_CYCCMD_STATES)
#error eEcatCycCmdState_SAFEOP>=EC_NUM_CYCCMD_STATES
#endif
#if (eEcatCycCmdState_OP>=EC_NUM_CYCCMD_STATES)
#error eEcatCycCmdState_OP>=EC_NUM_CYCCMD_STATES
#endif



/* EtherCat specific notify codes */
#define EC_NOTIFY_GENERIC                       0x00000000
#define EC_NOTIFY_ERROR                         0x00010000
#define EC_NOTIFY_MBOXRCV                       0x00020000
#define EC_NOTIFY_SCANBUS                       0x00030000
#define EC_NOTIFY_HOTCONNECT                    0x00040000        

#define EC_NOTIFY_STATECHANGED                  (EC_NOTIFY_GENERIC | 1)     /* EtherCAT operational state change */
#define EC_NOTIFY_ETH_LINK_CONNECTED            (EC_NOTIFY_GENERIC | 2)     /* Ethernet link (cable) connected */
#define EC_NOTIFY_SB_STATUS                     (EC_NOTIFY_GENERIC | 3)     /* ScanBus finished */ 
#define EC_NOTIFY_DC_STATUS                     (EC_NOTIFY_GENERIC | 4)     /* Distributed clocks initialized */

/* Distributed clocks (DC) */
#define EC_DC_SLV_SYNC_NTFY                     (EC_NOTIFY_GENERIC | 5)     /* DC Slave Synchronization deviation notification */
#define EC_NOTIFY_DC_SLV_SYNC                   (EC_NOTIFY_GENERIC | 5)     /* DC Slave Synchronization deviation notification */

/* Distributed Clocks Latching (DCL) */
#define EC_NOTIFY_DCL_STATUS                    (EC_NOTIFY_GENERIC | 8)     /* DCL initialized */
#define EC_NOTIFY_DCL_SLV_LATCH_EVT             (EC_NOTIFY_GENERIC | 9)     /* Single mode Slave Latch notification */
#define EC_NOTIFY_DCL_SLV_TIMER_READ            (EC_NOTIFY_GENERIC | 10)    /* Slave counter read notification (IOCTL/cont. mode) */

/* Update Slave States Result */
#define EC_NOTIFY_UPDATE_SLAVE_STATE_DONE       (EC_NOTIFY_GENERIC | 20)    /* "EC_IOCTL_INITIATE_UPDATE_ALL_SLAVE_STATE" done */

#define EC_NOTIFY_RAWCMD_DONE                   (EC_NOTIFY_GENERIC | 100)   /* Queue Raw Command Response Notification */

#ifdef INCLUDE_COE_PDO_SUPPORT
#define EC_NOTIFY_COE_TX_PDO                    (EC_NOTIFY_MBOXRCV | 1)     /* TxPDO notification */
#endif

/* errors */
#define EC_NOTIFY_CYCCMD_WKC_ERROR              (EC_NOTIFY_ERROR | 1)       /* cyclic command: working counter error */
#define EC_NOTIFY_MASTER_INITCMD_WKC_ERROR      (EC_NOTIFY_ERROR | 2)       /* master init command: working counter error */
#define EC_NOTIFY_SLAVE_INITCMD_WKC_ERROR       (EC_NOTIFY_ERROR | 3)       /* slave init command: working counter error */
#define EC_NOTIFY_EOE_MBXRCV_WKC_ERROR          (EC_NOTIFY_ERROR | 4)       /* EOE mbox receive: working counter error */
#define EC_NOTIFY_COE_MBXRCV_WKC_ERROR          (EC_NOTIFY_ERROR | 5)       /* COE mbox receive: working counter error */
#define EC_NOTIFY_FOE_MBXRCV_WKC_ERROR          (EC_NOTIFY_ERROR | 6)       /* FOE mbox receive: working counter error */
#define EC_NOTIFY_EOE_MBXSND_WKC_ERROR          (EC_NOTIFY_ERROR | 7)       /* EOE mbox send: working counter error */
#define EC_NOTIFY_COE_MBXSND_WKC_ERROR          (EC_NOTIFY_ERROR | 8)       /* COE mbox send: working counter error */
#define EC_NOTIFY_FOE_MBXSND_WKC_ERROR          (EC_NOTIFY_ERROR | 9)       /* FOE mbox send: working counter error */
#define EC_NOTIFY_FRAME_RESPONSE_ERROR          (EC_NOTIFY_ERROR |10)       /* Got no response on a sent Ethernet frame */
#define EC_NOTIFY_SLAVE_INITCMD_RESPONSE_ERROR  (EC_NOTIFY_ERROR |11)       /* Got no response on a sent ecat init command from slave */
#define EC_NOTIFY_MASTER_INITCMD_RESPONSE_ERROR (EC_NOTIFY_ERROR |12)       /* Got no response on a sent ecat master init command */
#define EC_NOTIFY_CMD_MISSING                   (EC_NOTIFY_ERROR |13)       /* Missing ecat command in received ethernet frame */
#define EC_NOTIFY_MBSLAVE_INITCMD_TIMEOUT       (EC_NOTIFY_ERROR |14)       /* Timeout when waiting for mailbox init command response */
#define EC_NOTIFY_NOT_ALL_DEVICES_OPERATIONAL   (EC_NOTIFY_ERROR |15)       /* Not all slave devices are in operational state when receiving cyclic frames */
#define EC_NOTIFY_ETH_LINK_NOT_CONNECTED        (EC_NOTIFY_ERROR |16)       /* Ethernet link (cable) not connected */
#define EC_NOTIFY_CYCCMD_TIMEOUT                (EC_NOTIFY_ERROR |17)       /* obsolete */
#define EC_NOTIFY_RED_LINEBRK                   (EC_NOTIFY_ERROR |18)       /* Redundancy: line break detected */
#define EC_NOTIFY_STATUS_SLAVE_ERROR            (EC_NOTIFY_ERROR |19)       /* At least one slave is in error state when receiving cyclic frames (BRD AL-STATUS) */
#define EC_NOTIFY_SLAVE_ERROR_STATUS_INFO       (EC_NOTIFY_ERROR |20)       /* Slave Error status information */
#define EC_NOTIFY_SLAVE_NOT_ADDRESSABLE         (EC_NOTIFY_ERROR |21)       /* Fixed station address lost (or slave missing) - FPRD to AL_STATUS failed  */
#define EC_NOTIFY_SOE_MBXRCV_WKC_ERROR          (EC_NOTIFY_ERROR |22)       /* SOE mbox receive: working counter error */
#define EC_NOTIFY_SOE_MBXSND_WKC_ERROR          (EC_NOTIFY_ERROR |23)       /* SOE mbox send: working counter error */
#define EC_NOTIFY_SOE_WRITE_ERROR               (EC_NOTIFY_ERROR |24)       /* SOE mbox write responded with an error */
#define EC_NOTIFY_MBSLAVE_COE_SDO_ABORT         (EC_NOTIFY_ERROR |25)       /* COE mbox SDO abort */
#define EC_NOTIFY_CLIENTREGISTRATION_DROPPED    (EC_NOTIFY_ERROR |26)       /* Client registration dropped, possibly call to ecatConfigureMaster by other thread (RAS) */
#define EC_NOTIFY_RED_LINEFIXED                 (EC_NOTIFY_ERROR |27)       /* Redundancy: line is repaired */
#define EC_NOTIFY_FOE_MBSLAVE_ERROR             (EC_NOTIFY_ERROR |28)       /* FOE mbox abort */
#define EC_NOTIFY_MBXRCV_INVALID_DATA           (EC_NOTIFY_ERROR |29)       /* invalid mail box data received */
#define EC_NUM_ERROR_NOTIFICATIONS                                29        /* number of error notifications */

/* ScanBus Notification */
#define EC_NOTIFY_SB_MISMATCH                   (EC_NOTIFY_SCANBUS | 2)     /* ScanBus mismatch */
#define EC_NOTIFY_SB_DUPLICATE_HC_NODE          (EC_NOTIFY_SCANBUS | 3)     /* ScanBus mismatch. A duplicate HC group was detected equal product code, vendor id 
                                                                               and alias address (or switch id)  */

/* Hot Connect Action Results */
/* #define EC_NOTIFY_HC_DETECTALLGROUPS           (EC_NOTIFY_HOTCONNECT | 1)  / * HC Detect All Groups done * / */
#define EC_NOTIFY_HC_DETECTADDGROUPS            (EC_NOTIFY_HOTCONNECT | 2)  /* HC Enhance Detect All Groups done */
#define EC_NOTIFY_HC_PROBEALLGROUPS             (EC_NOTIFY_HOTCONNECT | 3)  /* HC Probe All Groups done */
#define EC_NOTIFY_HC_TOPOCHGDONE                (EC_NOTIFY_HOTCONNECT | 4)  /* HC Topology Change done */

#define EC_NOTIFY_HC_SLAVE_PART                 (EC_NOTIFY_HOTCONNECT | 5)  /* ScanBus Slave Disappears */
#define EC_NOTIFY_HC_SLAVE_JOIN                 (EC_NOTIFY_HOTCONNECT | 6)  /* ScanBus Slave Appears */

/* cyclic error notification bit masks */
#define EC_CYC_ERR_MASK_NOT_ALL_DEVICES_OPERATIONAL     0x00000001      /* mask for cyclic EC_NOTIFY_NOT_ALL_DEVICES_OPERATIONAL notifications */
#define EC_CYC_ERR_MASK_STATUS_SLAVE_ERROR              0x00000002      /* mask for cyclic EC_NOTIFY_STATUS_SLAVE_ERROR notifications */
#define EC_CYC_ERR_MASK_CYCCMD_WKC_ERROR                0x00000004      /* mask for cyclic EC_NOTIFY_CYCCMD_WKC_ERROR notifications */
#define EC_CYC_ERR_MASK_UNEXPECTED_FRAME_RESPONSE       0x00000008      /* mask for cyclic EC_NOTIFY_FRAME_RESPONSE_ERROR notifications with error type eRspErr_UNEXPECTED */
#define EC_CYC_ERR_MASK_NO_FRAME_RESPONSE_ERROR         0x00000010      /* mask for cyclic EC_NOTIFY_FRAME_RESPONSE_ERROR notifications with error type eRspErr_NO_RESPONSE */
#define EC_ERR_MASK_ETH_LINK_NOT_CONNECTED              0x00000020      /* mask for EC_NOTIFY_ETH_LINK_NOT_CONNECTED notifications */
#define EC_CYC_ERR_MASK_ALL                             0xFFFFFFFF      /* mask for all cyclic error notifications */


/* CoE debug bit masks */
#define EC_COE_DBG_MASK_SDO_DOWNLOAD        0x00000001  /* mask for SDO download transfer debug support */
#define EC_COE_DBG_MASK_SDO_UPLOAD          0x00000002  /* mask for SDO upload transfer debug support */
#define EC_COE_DBG_MASK_GET_ODLIST          0x00000004  /* mask for CoE get object dictionary list transfer debug support */
#define EC_COE_DBG_MASK_GET_OBDESC          0x00000008  /* mask for CoE get object description transfer debug support */
#define EC_COE_DBG_MASK_GET_ENTRYDESC       0x00000010  /* mask for CoE get entry description transfer debug support */
#ifdef INCLUDE_COE_PDO_SUPPORT
#define EC_COE_DBG_MASK_RX_PDO              0x00000020  /* mask for RxPDO transfer debug support */
#define EC_COE_DBG_MASK_TX_PDO              0x00000040  /* mask for TxPDO transfer debug support */
#endif
#define EC_COE_DBG_MASK_ALL                 0x00000FFF  /* mask for all CoE transfers debug support */

/* FoE debug bit masks */
#define EC_FOE_DBG_MASK_FILE_DOWNLOAD       0x00001000  /* mask for FoE download transfer debug support */
#define EC_FOE_DBG_MASK_FILE_UPLOAD         0x00002000  /* mask for FoE download transfer debug support */
#define EC_FOE_DBG_MASK_ALL                 0x000FF000  /* mask for all FoE transfers debug support */

#define EC_MBX_DBG_MASK_ALL                 0x000FFFFF  /* mask for all CoE transfers debug support */


/* SoE debug bit masks */
#ifdef INCLUDE_SOE_SUPPORT
#define EC_SOE_DBG_MASK_IDN_WRITE        0x00000001  /* mask for SOE download transfer debug support */
#define EC_SOE_DBG_MASK_IDN_READ         0x00000002  /* mask for SOE upload transfer debug support */
#define EC_SOE_DBG_MASK_ALL              0x00000FFF  /* mask for all SoE transfers debug support */
#endif

/* EtherCAT commands */
typedef enum 
{
    EC_CMD_TYPE_NOP     = 0x00,
    EC_CMD_TYPE_APRD    = 0x01,
    EC_CMD_TYPE_APWR    = 0x02,
    EC_CMD_TYPE_APRW    = 0x03,
    EC_CMD_TYPE_FPRD    = 0x04,
    EC_CMD_TYPE_FPWR    = 0x05,
    EC_CMD_TYPE_FPRW    = 0x06,
    EC_CMD_TYPE_BRD     = 0x07,
    EC_CMD_TYPE_BWR     = 0x08,
    EC_CMD_TYPE_BRW     = 0x09,
    EC_CMD_TYPE_LRD     = 0x0A,
    EC_CMD_TYPE_LWR     = 0x0B,
    EC_CMD_TYPE_LRW     = 0x0C,
    EC_CMD_TYPE_ARMW    = 0x0D,
    EC_CMD_TYPE_FRMW    = 0x0E,
    EC_CMD_TYPE_EXT     = 0xFF,

    /* Borland C++ datatype alignment correction */
    EC_CMD_TYPE_BCppDummy   = 0xFFFFFFFF
} EC_CMD_TYPE;


/*-TYPEDEFS------------------------------------------------------------------*/

/* Synchronization modes to refresh and access to the IO process data */
typedef enum _EC_T_SYNCMODE
{
    eSyncMode_UNDEFINED        = 0,     /*< undefined */
    eSyncMode_USER_CTL         = 5,     /*< user controlled mode */

    /* Borland C++ datatype alignment correction */
    eSyncMode_BCppDummy   = 0xFFFFFFFF
} EC_T_SYNCMODE;

/* static master configuration */
typedef struct _EC_T_MASTER_CONFIG
{
    EC_T_DWORD  dwValPattern;                   /*< Validation pattern */
    EC_T_INT    nSlaveMultiplier;               /*< slave timer multiplier (based on master timer) */
    EC_T_DWORD  dwEcatCmdTimeout;               /*< timeout to send pending ethercat command frames */
    EC_T_DWORD  dwEcatCmdMaxRetries;            /*< maximum retries to send pending ethercat command frames */
    EC_T_DWORD  dwCycTimeout;                   /*< obsolete */
    EC_T_DWORD  dwCoeTimeout;                   /*< timeout sending CoE frames obsolete */
    EC_T_DWORD  dwEoeTimeout;                   /*< timeout sending EoE frames */
    EC_T_DWORD  dwFoeTimeout;                   /*< timeout sending FoE frames obsolete */
    EC_T_DWORD  dwFoeBusyTimeout;               /*< timeout FoE busy */
    EC_T_DWORD  dwMaxQueuedEthFrames;           /*< maximum number of queued Ethernet frames */
    EC_T_DWORD  dwMaxSlaveCmdPerFrame;          /*< maximum number of slave commands per frame */
    EC_T_DWORD  dwMaxQueuedCoeSlaves;           /*< maximum number of slaves queued to send CoE commands */
    EC_T_DWORD  dwMaxQueuedCoeCmds;             /*< maximum number of queued CoE commands */
    EC_T_DWORD  dwReserved;                     /*< obsolete */
    EC_T_DWORD  dwStateChangeDebug;             /*< Slave state change debugging level: Bit 0 = OsDbgMsg Bit 1 = Link-Msg */
    EC_T_DWORD  dwErrorMsgToLinkLayer;          /*< if EC_TRUE error messages are sent to the link layer */
    EC_T_DWORD  dwMaxSentQueuedFramesPerCyc;    /*< maximum number of queued frames to be sent after sending cyclic frames */
} EC_PACKED EC_T_MASTER_CONFIG;


/* EtherCAT slave properties */
typedef struct _EC_T_SLAVE_PROP
{
    EC_T_WORD   wStationAddress;        /*< slave node number (physical address), 
                                         *  set by XML configuration file (Config/Slave/Info/PhysAddr) */
    EC_T_WORD   wAutoIncAddr;           /*< auto increment address */
    EC_T_CHAR   achName[MAX_STD_STRLEN];/*< device name */
} EC_PACKED EC_T_SLAVE_PROP;


/* EtherCAT notify parameters */
typedef struct _EC_T_NOTIFYPARMS
{
    EC_T_VOID*      pCallerData;        /*< in  caller data */
    EC_T_BYTE*      pbyInBuf;           /*< in  input data buffer */
    EC_T_DWORD      dwInBufSize;        /*< in  size of input data buffer in byte */
    EC_T_BYTE*      pbyOutBuf;          /*< out output data buffer */
    EC_T_DWORD      dwOutBufSize;       /*< in  size of output data buffer in byte */
    EC_T_DWORD*     pdwNumOutData;      /*< out number of output data bytes stored in output data buffer */
} EC_PACKED EC_T_NOTIFYPARMS, *PEC_T_NOTIFYPARMS;

/* EtherCAT command WKC error descriptor */
typedef struct _EC_T_WKCERR_DESC
{
    EC_T_SLAVE_PROP SlaveProp;          /*< slave properties if applicable */
    EC_T_BYTE       byCmd;              /*< EtherCAT command where the error occured */
    EC_T_BYTE       byRsvd[3];          /*< Alignment */
    EC_T_DWORD      dwAddr;             /*< logical or physical (adp/ado) address */
    EC_T_WORD       wWkcSet;            /*< working counter set value */
    EC_T_WORD       wWkcAct;            /*< working counter actual value */
} EC_PACKED EC_T_WKCERR_DESC;

/* EtherCAT frame response error descriptor */
typedef enum _EC_T_FRAME_RSPERR_TYPE
{
    eRspErr_UNDEFINED       = 0,        /*< undefined */
    eRspErr_NO_RESPONSE     = 1,        /*< no response (timeout)            obsolete for acyclic frames*/
    eRspErr_WRONG_IDX       = 2,        /*< frame with wrong IDX received    obsolete for acyclic frames*/
    eRspErr_UNEXPECTED      = 3,        /*< unexpected frame received */
    eRspErr_FRAME_RETRY     = 4,        /*< ethernet frame has to be re-sent due to an timeout (only acyclic frames)  obsolete*/
    eRspErr_RETRY_FAIL      = 5,        /*< all retry mechanism fails to re-sent acyclic frames*/
    /* Borland C++ datatype alignment correction */
    eRspErr_BCppDummy   = 0xFFFFFFFF
} EC_T_FRAME_RSPERR_TYPE;

typedef struct _EC_T_FRAME_RSPERR_DESC
{
    EC_T_BOOL               bIsCyclicFrame;         /*< EC_TRUE if it is a frame containing cyclic commands */
    EC_T_FRAME_RSPERR_TYPE  EErrorType;             /*< type of error */
    EC_T_BYTE               byEcCmdHeaderIdxSet;    /*< Expected index value */
    EC_T_BYTE               byEcCmdHeaderIdxAct;    /*< Actual index value */
} EC_PACKED EC_T_FRAME_RSPERR_DESC;

/* EtherCAT init command response error descriptor */
typedef enum _EC_T_INITCMD_ERR_TYPE
{
    eInitCmdErr_UNDEFINED        = 0,    /*< undefined */
    eInitCmdErr_NO_RESPONSE      = 1,    /*< no response */
    eInitCmdErr_VALIDATION_ERR   = 2,    /*< validation error */
    eInitCmdErr_FAILED           = 3,    /*< init command failed, requested state could not be reached */

    /* Borland C++ datatype alignment correction */
    eInitCmdErr_BCppDummy       = 0xFFFFFFFF
} EC_T_INITCMD_ERR_TYPE;

typedef struct _EC_T_INITCMD_ERR_DESC
{
    EC_T_SLAVE_PROP       SlaveProp;                              /*< slave properties */
    EC_T_CHAR             achStateChangeName[MAX_SHORT_STRLEN];   /*< description of the current ecat state change */
    EC_T_INITCMD_ERR_TYPE EErrorType;                             /*< type of error */
} EC_PACKED EC_T_INITCMD_ERR_DESC;

/* EtherCAT missing command descriptor */
typedef struct _EC_T_MISSCMD_DESC
{
    EC_T_SLAVE_PROP SlaveProp;          /*< slave properties */
    EC_T_DWORD      dwCmdIndex;         /*< ethercat command/telegram index in ethernet frame that is missing */
    EC_T_BOOL       bMasterCmd;         /*< EC_TRUE if it is a master command or EC_FALSE if it is a slave command */
} EC_PACKED EC_T_MISSCMD_DESC;

/* EtherCAT Slave error status info descriptor */
typedef struct _EC_T_SLAVE_ERROR_INFO_DESC
{
    EC_T_SLAVE_PROP SlaveProp;          /*< slave properties */
    EC_T_WORD       wStatus;            /*< Slave Status (AL Status) */
    EC_T_WORD       wStatusCode;        /*< Error status code (AL Control Status) */
} EC_PACKED EC_T_SLAVE_ERROR_INFO_DESC;

/* EtherCAT Sdo abort */
typedef struct _EC_T_MBOX_SDO_ABORT_DESC
{
    EC_T_SLAVE_PROP SlaveProp;          /*< slave properties */
    EC_T_DWORD      dwErrorCode;        /*< error code EC_E_ */
    EC_T_WORD       wObjIndex;          /*< SDO object index */
    EC_T_BYTE       bySubIndex;         /*< SDO object sub index */
} EC_PACKED EC_T_MBOX_SDO_ABORT_DESC;

/* EtherCAT FoE error */
typedef struct _EC_T_MBOX_FOE_ABORT_DESC
{
    EC_T_SLAVE_PROP SlaveProp;          /*< slave properties */
    EC_T_DWORD      dwErrorCode;        /*< error code EC_E_ */
    EC_T_CHAR       achErrorString[MAX_STD_STRLEN];   /*< FOE error string */
} EC_PACKED EC_T_MBOX_FOE_ABORT_DESC;

/* Invalid mailbox data received error */
typedef struct _EC_T_MBXRCV_INVALID_DATA_DESC
{
    EC_T_SLAVE_PROP SlaveProp;          /*< slave properties */
} EC_PACKED EC_T_MBXRCV_INVALID_DATA_DESC;

/* Redundancy break/fixed notification */
typedef struct _EC_T_RED_CHANGE_DESC
{
    EC_T_WORD       wNumOfSlavesMain;    /*< Number of Slaves on Main Line */
    EC_T_WORD       wNumOfSlavesRed;     /*< Number of Slaves on Red Line  */
} EC_PACKED EC_T_RED_CHANGE_DESC;

/* Distributed clocks (DC) */
typedef struct _EC_T_DC_SYNC_NTFY_DESC
{
    EC_T_DWORD      IsInSync;           /*< EC_TRUE if all meant objects are in defined sync range, EC_FALSE otherwise */
    EC_T_DWORD      IsNegative;         /*< Deviation value is a negative value */
    EC_T_DWORD      dwDeviation;        /*< absolute deviation in nsecs (depending on notification nsec / wire or'ed value */
} EC_PACKED EC_T_DC_SYNC_NTFY_DESC;

typedef struct _EC_T_DC_BURSTCONFIG_REQ
{
    EC_T_DWORD	dwTotalBurstLength;
    EC_T_DWORD	dwBurstBulk;
} EC_PACKED EC_T_DC_BURSTCONFIG_REQ;

typedef struct _EC_T_DC_SHIFTSYSTIME_DESC
{
    EC_T_DWORD	dwCyclesToShift;    /*< Amount of cycles to apply shift */
    EC_T_INT        nShiftTime;         /*< Shift Time in nsecs (signed) */
} EC_PACKED EC_T_DC_SHIFTSYSTIME_DESC;

typedef  struct _EC_T_DC_STARTCYCSAFETY_DESC    /*< Start SYNC Cyc Safety in nsec (64Bit)*/
{
    EC_T_DWORD      dwStartCycSafetyLo;
    EC_T_DWORD      dwStartCycSafetyHi;
}   EC_T_DC_STARTCYCSAFETY_DESC;
           
typedef struct _EC_T_SB_STATUS_NTFY_DESC
{
    EC_T_DWORD      dwResultCode;       /*< Result of scanbus */
    EC_T_DWORD      dwSlaveCount;       /*< Number of slaves handles during scanbus */
} EC_PACKED EC_T_SB_STATUS_NTFY_DESC;

typedef struct _EC_T_SB_MISMATCH_DESC
{
    /* Previous */
    EC_T_WORD           wPrevFixedAddress;      /*< Previous fixed address */
    EC_T_WORD           wPrevPort;              /*< Previous port */
    EC_T_WORD           wPrevAIncAddress;       /*< Previous Auto-Inc address */
                                                
    /* error or mismatching slave */                        
    EC_T_WORD           wBusAIncAddress;        /*< Bus Auto-Inc address */
    EC_T_DWORD          dwBusVendorId;          /*< Bus Vendor ID */
    EC_T_DWORD          dwBusProdCode;          /*< Bus Product code */
    EC_T_DWORD          dwBusRevisionNo;        /*< Bus Revision Number */
    EC_T_DWORD          dwBusSerialNo;          /*< Bus Serial Number */
                                                
    EC_T_WORD           wBusFixedAddress;       /*< Bus Fixed Address */
    EC_T_WORD           wBusAliasAddress;       /*< Bus Alias Address */
                        
    EC_T_WORD           wCfgFixedAddress;       /*< Config Fixed Address */
    EC_T_WORD           wCfgAIncAddress;        /*< Config Auto-Increment Address */
                        
    EC_T_DWORD          dwCfgVendorId;          /*< Config Vendor ID */
    EC_T_DWORD          dwCfgProdCode;          /*< Config Product code */
    EC_T_DWORD          dwCfgRevisionNo;        /*< Config Revision Number */
    EC_T_DWORD          dwCfgSerialNo;          /*< Config Serial Number */
                                                
    EC_T_BOOL           bIdentValidationError;  /*< HotConnect Identification command validation error */

    EC_T_WORD           oIdentCmdHdr[5];        /*< HotConnect Identification command header (if bIdentValidationError) */
    EC_T_DWORD          dwCmdData;              /*< First DWORD of Data portion in identify command (received data) */
    EC_T_DWORD          dwCmdVMask;             /*< First DWORD of Validation mask in identify command */
    EC_T_DWORD          dwCmdVData;             /*< First DWORD of Validation data in identify command (expected data) */

} EC_PACKED EC_T_SB_MISMATCH_DESC;

typedef struct _EC_T_HC_DETECTALLGROUP_NTFY_DESC
{
    EC_T_DWORD      dwResultCode;       /*< Result of Group detection */
    EC_T_DWORD      dwGroupCount;       /*< Number of Groups */
    EC_T_DWORD      dwGroupsPresent;    /*< Number of connected groups */
    EC_T_DWORD      dwGroupMask;        /*< Bitmask of first 32 Groups 1 = present, 0 = absent */
} EC_PACKED EC_T_HC_DETECTALLGROUP_NTFY_DESC;

typedef struct _EC_T_DC_LATCH_EVENT_DESC
{
    EC_T_DWORD	dwSlaveId;              /*< PhysAddr */
    EC_T_DWORD	dwLatchIdent;           /*< Identification which timer fields are latched */
} EC_PACKED EC_T_DC_LATCH_EVENT_DESC;

typedef struct _EC_T_DC_LATCH_MODE{
    EC_T_DWORD	dwImplicitQueryEnable;  /*< EC_TRUE:    Latch Timer registers are queried by master implicitly
                                         *  EC_FALSE:   Application has to read timers by calling 
                                         *              ecatIoControl - EC_IOCTL_DC_LATCH_REQ_LTIMVALS.
                                         */

    EC_T_DWORD	dwDoLatchNotifyEvt;     /*< Only used if dwImplicitQueryEnable = EC_TRUE
                                         *  EC_TRUE:    Latch Notification Event ( EC_NOTIFY_DCL_SLV_LATCH_EVT )
                                         *              is raised, although application does not need to react to it
                                         *  EC_FALSE:   Single Latch Notification is suppressed.
                                         */
} EC_PACKED EC_T_DC_LATCH_MODE;



typedef struct _EC_T_DCL_SLV_TIMER_NTFY_DESC{
    EC_T_DWORD	dwSlaveId;              /*< PhysAddr */
    EC_T_DWORD	dwLatchIdent;           /*< Identification which timer fields are latched */
    EC_T_DWORD	dwLatch0PosLo;          /*< Latch 0 Positive Low Part, only valid if ident is set */
    EC_T_DWORD	dwLatch0PosHi;          /*< Latch 0 Positive High Part, only valid if ident is set */
    EC_T_DWORD	dwLatch0NegLo;          /*< Latch 0 Negative Low Part, only valid if ident is set */ 
    EC_T_DWORD	dwLatch0NegHi;          /*< Latch 0 Negative High Part, only valid if ident is set */
    EC_T_DWORD	dwLatch1PosLo;          /*< Latch 1 Positive Low Part, only valid if ident is set */ 
    EC_T_DWORD	dwLatch1PosHi;          /*< Latch 1 Positive High Part, only valid if ident is set */
    EC_T_DWORD	dwLatch1NegLo;          /*< Latch 1 Negative Low Part, only valid if ident is set */ 
    EC_T_DWORD	dwLatch1NegHi;          /*< Latch 1 Negative High Part, only valid if ident is set */
} EC_PACKED EC_T_DCL_SLV_TIMER_NTFY_DESC;
                                                   
typedef struct _EC_T_RAWCMDRESPONSE_NTFY_DESC
{
    EC_T_DWORD	dwInvokeId;
    EC_T_DWORD	dwResult;
    EC_T_DWORD	dwWkc;
    EC_T_DWORD	dwCmdIdx;
    EC_T_DWORD	dwAddr;
    EC_T_DWORD	dwLength;
    EC_T_BYTE*      pbyData;
} EC_PACKED EC_T_RAWCMDRESPONSE_NTFY_DESC;
                                                   
#ifdef INCLUDE_COE_PDO_SUPPORT
typedef struct _EC_T_TX_PDO_NTFY_DESC
{
    EC_T_DWORD  wPhysAddr;          /* physical slave address */
    EC_T_DWORD  dwNumber;           /* PDO number */
    EC_T_DWORD  wLen;               /* PDO size */
    EC_T_BYTE*  pbyData;
} EC_PACKED EC_T_TX_PDO_NTFY_DESC;
#endif

typedef struct _EC_T_SLAVE_LINKMSG_DESC
{
    EC_T_DWORD  dwSlaveId;          /* Slave Id */
    EC_T_BOOL   bEnableLogging;     /* EC_TRUE=> Enable LinkMsgs, EC_FALSE=> Disable */
} EC_PACKED EC_T_SLAVE_LINKMSG_DESC;

/* EtherCAT state */
typedef enum _EC_T_STATE
{
    eEcatState_UNKNOWN  = 0,                        /*< unknown */
    eEcatState_INIT     = 1,                        /*< init */
    eEcatState_PREOP    = 2,                        /*< pre-operational */
    eEcatState_SAFEOP   = 4,                        /*< safe operational */
    eEcatState_OP       = 8,                        /*< operational */
    
    /* Borland C++ datatype alignment correction */
    eEcatState_BCppDummy   = 0xFFFFFFFF
} EC_T_STATE;                                       

/* EtherCAT state change */                         
typedef struct _EC_T_STATECHANGE                    
{                                                   
    EC_T_STATE    oldState;                         /*< old state */
    EC_T_STATE    newState;                         /*< new state */
} EC_PACKED EC_T_STATECHANGE;                                 


/* List of slaves */
#define MAX_NUM_OF_SLAVES_WITH_ERR_STATE    128
typedef struct _EC_T_SLAVE_LIST_DESC
{                                                   
    EC_T_DWORD    dwListLen;                         /*< number of entrys in list */
    EC_T_DWORD    adwSlaveID[MAX_NUM_OF_SLAVES_WITH_ERR_STATE];  /*< array of Slave IDs */
} EC_PACKED EC_T_SLAVE_LIST_DESC;

typedef struct _EC_T_HC_SLAVE_CHANGE_DESC
{
    EC_T_SLAVE_PROP     SlaveProp;
} EC_PACKED EC_T_HC_SLAVE_CHANGE_DESC;

/* EtherCAT error notification descriptor */
typedef struct _EC_T_NOTIFICATION_DESC
{
    union _EC_T_NOTIFICATION_PARM
    {                
        EC_T_DWORD              StatusCode;             /* Generic status code only notification */
        EC_T_DC_SYNC_NTFY_DESC  SyncNtfyDesc;           /* DC Master / Slave Sync Notification descriptor */
        EC_T_RAWCMDRESPONSE_NTFY_DESC
                                RawCmdRespNtfyDesc;     /* Queue Raw Cmd Response notification descriptor */
        EC_T_SB_STATUS_NTFY_DESC
                                ScanBusNtfyDesc;        /* Scanbus Result notification descriptor */

        EC_T_SB_MISMATCH_DESC   ScanBusMismatch;        /* Scan Bus Mismatch notification descriptor */

        EC_T_DC_LATCH_EVENT_DESC
                                LatchEvtNtfyDesc;       /* Latch Event notification */
        EC_T_DCL_SLV_TIMER_NTFY_DESC
                                LatchSlvTimerNtfyDesc;  /* Detailed latching information notification */
        EC_T_STATECHANGE        StatusChngNtfyDesc;     /* Status changed notification */
#ifdef INCLUDE_COE_PDO_SUPPORT
        EC_T_TX_PDO_NTFY_DESC   TxPdoNtfyDesc;          /* TxPDO transfer notification */
#endif
        EC_T_SLAVE_LIST_DESC    SlaveListDesc;          /* Contains a list of Slave IDs */
        EC_T_HC_DETECTALLGROUP_NTFY_DESC
                                HCDetAllGrps;           /* HC Group Detection */
        EC_T_HC_SLAVE_CHANGE_DESC
                                HCSlvChgDesc;           /* HotConnect Slave State Change */
    }desc;
} EC_PACKED EC_T_NOTIFICATION_DESC;

/* EtherCAT error notification descriptor */
typedef struct _EC_T_ERROR_NOTIFICATION_DESC
{
    EC_T_DWORD  dwNotifyErrorCode;                  /* notification code */
    EC_T_CHAR   achErrorInfo[MAX_ERRINFO_STRLEN];   /* additional error information string (NULL terminated) */
    union _EC_T_ERROR_NOTIFICATION_PARM
    {                                               
        EC_T_WKCERR_DESC              WkcErrDesc;            /* WKC error descriptor */
        EC_T_FRAME_RSPERR_DESC        FrameRspErrDesc;       /* Frame response error descriptor */
        EC_T_INITCMD_ERR_DESC         InitCmdErrDesc;        /* Master/Slave init command error descriptor */
        EC_T_MISSCMD_DESC             MissCmdDesc;           /* EtherCAT missing command descriptor */
        EC_T_SLAVE_ERROR_INFO_DESC    SlaveErrInfoDesc;      /* Slave Error Info Descriptor */
        EC_T_MBOX_SDO_ABORT_DESC      SdoAbortDesc;          /* SDO Abort */
        EC_T_RED_CHANGE_DESC          RedChangeDesc;         /* Redundancy Descriptor */
        EC_T_MBOX_FOE_ABORT_DESC      FoeErrorDesc;          /* FOE error code and string */
        EC_T_MBXRCV_INVALID_DATA_DESC MbxRcvInvalidDataDesc; /* Invalid mailbox data received descriptor */
    } desc;
} EC_PACKED EC_T_ERROR_NOTIFICATION_DESC;

typedef enum _EC_T_CNF_TYPE
{
    eCnfType_Unknown    = 0,
    eCnfType_Filename   = 1,
    eCnfType_Data       = 2,
    eCnfType_Hilscher   = 3,
    /* Borland C++ datatype alignment correction */
    eCnfType_BCppDummy   = 0xFFFFFFFF
} EC_T_CNF_TYPE;

typedef EC_T_VOID (*EC_T_PFMEMREQ)(EC_T_PVOID pvContext, EC_T_PBYTE* ppbyPDData);
typedef EC_T_VOID (*EC_T_PFMEMREL)(EC_T_PVOID pvContext);

/* Descriptor for EC_IOCTL_REGISTER_PDMEMORYPROVIDER */
typedef struct _EC_T_MEMPROV_DESC
{
    EC_T_PVOID      pvContext;                      /* Context pointer. This pointer is used each time one of
                                                     * the callback functions is called. 
                                                     */
    EC_T_BOOL       bUseMasterShadowInUserSyncNowaitMode;
                                                    /* This flag determines whether the master shall use an 
                                                     * internally allocated shadow memory where new input 
                                                     * process data will be stored when received from 
                                                     * the link layer. Only valid in 
                                                     * eSyncMode_USER_SYNC_NOWAIT.
                                                     */
    EC_T_PBYTE      pbyPDOutData;                   /* Pointer to the default output process data buffer */
    EC_T_DWORD      dwPDOutDataLength;              /* Length of the output process data buffer */

    EC_T_PBYTE      pbyPDInData;                    /* Pointer to the default input process data buffer */
    EC_T_DWORD      dwPDInDataLength;               /* Length of the input  process data buffer */

    EC_T_PFMEMREQ   pfPDOutDataReadRequest;         /* This function will be called by the master within the 
                                                     * process data transfer cycle prior to read data.
                                                     */
    EC_T_PFMEMREL   pfPDOutDataReadRelease;         /* This function will be called after all data was read 
                                                     * from output process data buffer.
                                                     */

    EC_T_PFMEMREQ   pfPDInDataWriteRequest;         /* This function will be called by the master within the 
                                                     * process data transfer cycle prior to write data.
                                                     */
    EC_T_PFMEMREL   pfPDInDataWriteRelease;         /* This function will be called after all data was written
                                                     * to input process data buffer.
                                                     */
} EC_PACKED EC_T_MEMPROV_DESC, *EC_PT_MEMPROV_DESC;

typedef struct _EC_T_SB_SLAVEINFO_DESC
{
    EC_T_DWORD  dwScanBusStatus;                    /* Status during last Bus Scan */
    
    EC_T_DWORD	dwVendorId;                         /* Vendor Identification */
    EC_T_DWORD	dwProductCode;                      /* Product Code */
    EC_T_DWORD	dwRevisionNumber;                   /* Revision Number */
    EC_T_DWORD	dwSerialNumber;                     /* Serial Number */
} EC_PACKED EC_T_SB_SLAVEINFO_DESC, *EC_PT_SB_SLAVEINFO_DESC;

typedef enum _EC_T_eEEPENTRY
{
    eEEP_VendorId	        = ESC_SII_REG_VENDORID,
    eEEP_ProductCode        = ESC_SII_REG_PRODUCTCODE,
    eEEP_RevisionNumber     = ESC_SII_REG_REVISIONNUMBER,
    eEEP_SerialNumber       = ESC_SII_REG_SERIALNUMBER,

    /* Borland C++ datatype alignment correction */
    eEEP_BCppDummy          = 0xFFFFFFFF
} EC_T_eEEPENTRY;

#define EEPENTRY_VENDORID       ((EC_T_DWORD)0)
#define EEPENTRY_PRODUCTCODE    ((EC_T_DWORD)1)
#define EEPENTRY_REVISIONNO     ((EC_T_DWORD)2)
#define EEPENTRY_SERIALNO       ((EC_T_DWORD)3)

#define BT_CHECK_EEPENTRY_VENDORID      EC_TRUE
#define BT_CHECK_EEPENTRY_PRODUCTCODE   EC_TRUE
#define BT_CHECK_EEPENTRY_REVISIONNO    EC_FALSE
#define BT_CHECK_EEPENTRY_SERIALNO      EC_FALSE

#define EEPENTRYTOINDEX(dwEnum) \
    ((dwEnum==eEEP_VendorId)?EEPENTRY_VENDORID: \
     ((dwEnum==eEEP_ProductCode)?EEPENTRY_PRODUCTCODE: \
      ((dwEnum==eEEP_RevisionNumber)?EEPENTRY_REVISIONNO: \
       ((dwEnum==eEEP_SerialNumber)?EEPENTRY_SERIALNO: 0 \
       ))))

#define EEPINDEXTOENTRY(dwIndex) \
    ((dwIndex==EEPENTRY_VENDORID)?eEEP_VendorId: \
     ((dwIndex==EEPENTRY_PRODUCTCODE)?eEEP_ProductCode: \
      ((dwIndex==EEPENTRY_REVISIONNO)?eEEP_RevisionNumber: \
       ((dwIndex==EEPENTRY_SERIALNO)?eEEP_SerialNumber: eEEP_BCppDummy \
       ))))

typedef struct _EC_T_SCANBUS_PROP_DESC
{
    EC_T_eEEPENTRY  eEEPROMEntry;
    EC_T_DWORD      dwVerify;
} EC_PACKED EC_T_SCANBUS_PROP_DESC, *EC_PT_SCANBUS_PROP_DESC;

typedef enum _EC_T_eSBSlaveInfoType
{
    sbsit_unknown	    = 0,
    sbsit_bustopology   = 1,
    sbsit_configuration = 2,

    /* Borland C++ datatype alignment correction */
    sbsit_BCppDummy     = 0xFFFFFFFF
} EC_T_eSBSlaveInfoType;

typedef struct _EC_T_SB_SLAVEINFO_EEP_REQ_DESC
{
    EC_T_eSBSlaveInfoType       eSbSlaveInfoType;
    EC_T_DWORD                  wAutoIncAddress;
    EC_T_eEEPENTRY              eEEPROMEntry;
} EC_T_SB_SLAVEINFO_EEP_REQ_DESC;

typedef struct _EC_T_SB_SLAVEINFO_EEP_RES_DESC
{
    EC_T_DWORD                  dwScanBusStatus;
    EC_T_eEEPENTRY              eEEPROMEntry;
    EC_T_DWORD                  dwEEPROMValue;
} EC_PACKED EC_T_SB_SLAVEINFO_EEP_RES_DESC;

typedef enum _EC_T_eINFOENTRY
{
    eie_unknown         =  0,       /** nothing / invalid */
    eie_pdoffs_in       =  1,       /** get process data offset of Input data (in Bits) */
    eie_pdsize_in       =  2,       /** get process data size of Input Data (in Bits) */
    eie_pdoffs_out      =  3,       /** get process data offset of Output data (in Bits) */
    eie_pdsize_out      =  4,       /** get process data size of Output Data (in Bits) */
    eie_phys_address    =  5,       /** get slave phys Address */
    eie_portstate       =  6,       /** get port link state (SB Instance) */
    eie_dcsupport       =  7,       /** slave does support DC */
    eie_dc64support     =  8,       /** slave does support 64Bit DC */
    eie_alias_address   =  9,       /** get slave alias address */
    eie_cfgphy_address  = 10,       /** get slave configured phys address */
    eie_device_name     = 11,       /** get slave name from configuration */
    eie_ismailbox_slave = 12,       /** get whether slave support mailboxes */
    eie_pdoffs_in2      = 21,       /** get process data offset of Input data (in Bits) */
    eie_pdsize_in2      = 22,       /** get process data size of Input Data (in Bits) */
    eie_pdoffs_out2     = 23,       /** get process data offset of Output data (in Bits) */
    eie_pdsize_out2     = 24,       /** get process data size of Output Data (in Bits) */
    eie_pdoffs_in3      = 31,       /** get process data offset of Input data (in Bits) */
    eie_pdsize_in3      = 32,       /** get process data size of Input Data (in Bits) */
    eie_pdoffs_out3     = 33,       /** get process data offset of Output data (in Bits) */
    eie_pdsize_out3     = 34,       /** get process data size of Output Data (in Bits) */
    eie_pdoffs_in4      = 41,       /** get process data offset of Input data (in Bits) */
    eie_pdsize_in4      = 42,       /** get process data size of Input Data (in Bits) */
    eie_pdoffs_out4     = 43,       /** get process data offset of Output data (in Bits) */
    eie_pdsize_out4     = 44,       /** get process data size of Output Data (in Bits) */
    
    eie_mbx_outsize     = 45,       /** get out mailbox 1 size */
    eie_mbx_insize      = 46,       /** get in mailbox 1 size */
    eie_mbx_outsize2    = 47,       /** get out mailbox 2 size */
    eie_mbx_insize2     = 48,       /** get in mailbox 2 size */

    eie_isoptional      = 49,       /** is slave optional */
    eie_ispresent       = 50,       /** is slave present on bus */

    eie_esctype         = 51,       /** Type of ESC controller */
    /* Borland C++ datatype alignment correction */
    eie_BCppDummy       = 0xFFFFFFFF
} EC_T_eINFOENTRY;

typedef struct _EC_T_SB_SLAVEINFO_REQ_DESC
{
    EC_T_eINFOENTRY eInfoEntry;
    EC_T_WORD       wAutoIncAddress;
} EC_PACKED EC_T_SB_SLAVEINFO_REQ_DESC;

typedef struct _EC_T_SB_SLAVEINFO_RES_DESC
{
    EC_T_eINFOENTRY eInfoEntry;
    EC_T_DWORD      dwInfoLength;
    EC_T_PBYTE      pbyInfo;
} EC_PACKED EC_T_SB_SLAVEINFO_RES_DESC;

typedef struct _EC_T_SB_RESIZEMEMORYPOOL
{
    EC_T_DWORD      dwNewPoolSize;
    EC_T_BOOL       bForceResize;
} EC_PACKED EC_T_SB_RESIZEMEMORYPOOL;

typedef struct _EC_T_DCL_ENABLE_DESC
{
    EC_T_DWORD	dwVerifySyncLatchConfiguration;
    EC_T_DWORD	dwDCLInitTimeout;
} EC_PACKED EC_T_DCL_ENABLE_DESC;


/* descriptor for EC_IOCTL_GET_CYCLIC_CONFIG_INFO call */
typedef struct _EC_T_CYC_CONFIG_DESC
{
    EC_T_DWORD      dwNumCycEntries;    /* total number of cyclic entries */
    EC_T_DWORD      dwTaskId;           /* task id of selected cyclic entry */
    EC_T_DWORD      dwPriority;         /* priority of selected cyclic entry */
    EC_T_DWORD      dwCycleTime;        /* cycle time of selected cyclic entry */
} EC_PACKED EC_T_CYC_CONFIG_DESC;

typedef EC_T_DWORD (*EC_PF_TIMESTAMP)(              EC_T_PVOID      pCallerData,
                                                    EC_T_DWORD*     pdwHostTimeLo   );
                                                    
/* Structure carrying instantaneous values for Master Sync */
typedef struct _EC_T_ACTUALVAL
{
    EC_T_DWORD	dwBeginUpdateCnt;               /*< out call counter incremented each time structure
                                                     *      update is started */
    EC_T_DWORD      dwEndUpdateCnt;                 /*< out call counter incremented each time structure
                                                     *      update is finished */
    EC_T_DWORD	dwBusSyncFrameSendTimeLo;       /*< out Host stamped time (result from pfTimeStamp) LSDW */
    EC_T_DWORD	dwBusTimeHi;                    /*< out Bus time (result from ARMW) MSDW */
    EC_T_DWORD	dwBusTimeLo;                    /*< out Bus time (result from ARMW) LSDW */
    EC_T_DWORD	dwBusSyncFramePostSendTimeLo;   /*< out Host stamped time (result from pfTimeStamp) LSDW */
    EC_T_DWORD	dwSyncPulseGridOffsetHi;        /*< out Sync Pulse Offset (initial grid offset) */
    EC_T_DWORD	dwSyncPulseGridOffsetLo;        /*< out Sync Pulse Offset (initial grid offset) */
    EC_T_DWORD	dwSyncPulseLength;              /*< out Pulse length of clock master slave in nsec */
    EC_T_DWORD	dwSyncPeriodLength;             /*< out Time between two consecutive SYNC0 signals of clock master slave in nsec. */
    EC_T_DWORD	dwSyncPulseSecOffset;           /*< out Time between SYNC1 and SYNC0 signal of clock master slave in nsec. */
    EC_T_DWORD      dwLastTimeStampResult;          /*< out Last result of call to registered callback 
                                                     *      function EC_PF_TIMESTAMP */
    EC_T_BOOL       bSlavesInSync;                  /*< out Slaves are in sync = EC_TRUE; out of sync = EC_FALSE; */
} EC_PACKED EC_T_ACTUALVAL;

/* Controller client register parameters */
typedef struct _EC_T_REGISTER_TSPARMS
{
    EC_T_VOID*	pCallerData;                    /*< in  used by all callback functions */  
    EC_PF_TIMESTAMP	pfTimeStamp;                    /*< in  timestamp callback function pointer */
    EC_T_DWORD	dwUpdateMultiplier;             /*< in  Interval multiplier */
    EC_T_DWORD	dwEnableHardRealtime;           /*< in  enable hard real time in link layer */
    EC_T_DWORD	aReserved[10];                  /*< in  reserved for future use */
} EC_PACKED EC_T_REGISTER_TSPARMS;

/* Controller client register result */
typedef struct _EC_T_REGISTER_TSRESULTS
{
    EC_T_DWORD          dwHardRealtimeEnabled;      /*< out Hard real time is used */
    EC_T_ACTUALVAL*     pActualValues;              /*< out pointer to data carrying current data */
} EC_PACKED EC_T_REGISTER_TSRESULTS;

typedef struct _EC_T_LINKLAYER_MODE_DESC
{
    EC_T_LINKMODE	eLinkMode;                      /*< out Linkmode of LL */
    EC_T_LINKMODE	eLinkModeRed;                   /*< out Linkmode of RedLL */
} EC_PACKED EC_T_LINKLAYER_MODE_DESC;

typedef struct _EC_T_SLVSTATISTICS_DESC
{
    EC_T_WORD       wRxErrorCounter[4];             /*< out RX Error Counters per Slave Port */
    EC_T_BYTE       byFwdRxErrorCounter[4];         /*< out Forwarded RX Error Counters per Slave Port */
    EC_T_BYTE       byEcatProcUnitErrorCounter;     /*< out EtherCAT Processing unit error counter */
    EC_T_BYTE       byPDIErrorCounter;              /*< out PDI Error counter */
    EC_T_WORD       wReserved;                      /*< out Reserved for alignment issues */
    EC_T_BYTE       byLostLinkCounter[4];           /*< out Lost Link Counter per Slave Port */
} EC_PACKED EC_T_SLVSTATISTICS_DESC;

typedef struct _EC_T_SLAVE_PORT_DESC
{
    EC_T_WORD       wSlaveAddress;
    EC_T_WORD       wPortNumber;
} EC_PACKED EC_T_SLAVE_PORT_DESC;

typedef enum _EC_T_EHOTCONNECTMODE
{
    echm_unknown        = 0x0000,
    echm_manual         = 0x0001,
    echm_automatic      = 0x0002,
    echm_automan_mask   = 0x000f,
    echm_borderclose    = 0x0010,

    echm_BCppDummy =   0xFFFFFFFF,
} EC_T_EHOTCONNECTMODE;

/* callback pointer for RX frame */ 
typedef EC_T_VOID (*EC_T_PF_RXFRAME_CB)(EC_T_VOID*);



/*-PACK SETTINGS-------------------------------------------------------------*/
#if defined  __GNUC__   /* GNU */

#undef  EC_PACKED

#if (defined __RCX__)
#pragma pack ()
#endif

#elif defined _MSC_VER  /* MICROSOFT */

#undef  EC_PACKED
#pragma pack (pop)

#elif defined __MET__

#undef EC_PACKED
/*#pragma pack (pop)*/
#pragma Pop_align_members()

#else
#endif

/*-FORWARD DECLARATIONS------------------------------------------------------*/

#ifdef __cplusplus
extern "C"
{
#endif

EC_T_CHAR*  ecatGetTextInt(                EC_T_WORD               wTextId         );

#ifdef __cplusplus
} /* extern "C" */
#endif


#endif /* INC_ECINTERFACECOMMON */

/*-END OF SOURCE FILE--------------------------------------------------------*/

